<?php

if (!defined('ABSPATH')) {
    exit;
}

final class GP_Plugin_Manager
{
    private GP_Repository $repo;
    private GP_Access $access;

    private string $update_cache_key;
    private int $update_cache_ttl;

    /**
     * @param array<string,mixed> $config
     */
    public function __construct(GP_Repository $repo, GP_Access $access, array $config = [])
    {
        $this->repo = $repo;
        $this->access = $access;

        $this->update_cache_key = isset($config['update_cache_key']) ? (string) $config['update_cache_key'] : GP_Get_Plugin::UPDATE_CACHE_KEY;
        $this->update_cache_ttl = isset($config['update_cache_ttl']) ? (int) $config['update_cache_ttl'] : GP_Get_Plugin::UPDATE_CACHE_TTL;
        if ($this->update_cache_ttl < 1) {
            $this->update_cache_ttl = 3600;
        }
    }

    public function invalidate_caches(): void
    {
        delete_transient($this->update_cache_key);
    }

    /**
     * @return array<string, array<int, array<string,mixed>>>
     */
    public function installed_plugins_map(): array
    {
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        $all = get_plugins();
        $map = [];

        foreach ($all as $plugin_file => $data) {
            $dir = dirname((string) $plugin_file);
            if ($dir === '.' || $dir === '') {
                continue;
            }

            if (!isset($map[$dir])) {
                $map[$dir] = [];
            }

            $map[$dir][] = [
                'plugin_file' => (string) $plugin_file,
                'name'        => isset($data['Name']) ? (string) $data['Name'] : '',
                'version'     => isset($data['Version']) ? (string) $data['Version'] : '',
                'active'      => function_exists('is_plugin_active') ? (bool) is_plugin_active($plugin_file) : false,
            ];
        }

        return $map;
    }

    /**
     * @param array<int, array<string,mixed>> $installed_candidates
     */
    public function pick_installed_plugin_file(string $slug, array $installed_candidates, string $remote_name = ''): string
    {
        if (empty($installed_candidates)) {
            return '';
        }

        if ($remote_name !== '') {
            foreach ($installed_candidates as $c) {
                if (!is_array($c)) {
                    continue;
                }
                $n = isset($c['name']) ? (string) $c['name'] : '';
                if ($n !== '' && strcasecmp($n, $remote_name) === 0) {
                    return isset($c['plugin_file']) ? (string) $c['plugin_file'] : '';
                }
            }
        }

        foreach ($installed_candidates as $c) {
            if (!is_array($c)) {
                continue;
            }
            $pf = isset($c['plugin_file']) ? (string) $c['plugin_file'] : '';
            if ($pf === $slug . '/' . $slug . '.php') {
                return $pf;
            }
        }

        $first = $installed_candidates[0];
        return isset($first['plugin_file']) ? (string) $first['plugin_file'] : '';
    }

    public function get_update_count(): int
    {
        $access = $this->access->check_access(false);
        if (empty($access['allowed'])) {
            return 0;
        }

        $cached = get_transient($this->update_cache_key);

        $manifest = $this->repo->get_manifest_data(false, 8);
        if (!is_array($manifest) || !empty($manifest['error']) || empty($manifest['plugins']) || !is_array($manifest['plugins'])) {
            if (is_array($cached) && isset($cached['count']) && is_numeric($cached['count'])) {
                return (int) $cached['count'];
            }
            return 0;
        }

        $gen = isset($manifest['generated_at']) ? (string) $manifest['generated_at'] : '';

        if (is_array($cached) && isset($cached['count'], $cached['generated_at']) && (string) $cached['generated_at'] === $gen) {
            return (int) $cached['count'];
        }

        $installed_map = $this->installed_plugins_map();
        $count = 0;

        foreach ($manifest['plugins'] as $row) {
            if (!is_array($row)) {
                continue;
            }

            $slug   = isset($row['slug']) ? (string) $row['slug'] : '';
            $name   = isset($row['name']) ? (string) $row['name'] : '';
            $latest = isset($row['latest']) ? (string) $row['latest'] : '';

            if ($slug === '' || $latest === '') {
                continue;
            }
            if (!isset($installed_map[$slug]) || empty($installed_map[$slug])) {
                continue;
            }

            $cands = $installed_map[$slug];
            $pf = $this->pick_installed_plugin_file($slug, $cands, $name);
            if ($pf === '') {
                continue;
            }

            $installed_ver = '';
            foreach ($cands as $c) {
                if (!is_array($c)) {
                    continue;
                }
                if (isset($c['plugin_file']) && (string) $c['plugin_file'] === $pf) {
                    $installed_ver = isset($c['version']) ? (string) $c['version'] : '';
                    break;
                }
            }

            if (GP_Utils::is_update_available($installed_ver, $latest)) {
                $count++;
            }
        }

        set_transient($this->update_cache_key, [
            'count'        => (int) $count,
            'generated_at' => $gen,
        ], $this->update_cache_ttl);

        return (int) $count;
    }

    /**
     * @return array{ok:bool, fs: mixed, error: string}
     */
    private function ensure_filesystem(): array
    {
        require_once ABSPATH . 'wp-admin/includes/file.php';

        $ok = WP_Filesystem();
        if (!$ok || empty($GLOBALS['wp_filesystem'])) {
            return ['ok' => false, 'fs' => null, 'error' => 'WP_Filesystem could not initialize (check permissions / FS_METHOD).'];
        }

        return ['ok' => true, 'fs' => $GLOBALS['wp_filesystem'], 'error' => ''];
    }

    /**
     * @param array<string,mixed> $remote
     * @return array{ok:bool, msg:string}
     */
    public function install_or_update_one(string $slug, string $version, array $remote, bool $activate_after, bool $preserve_active): array
    {
        require_once ABSPATH . 'wp-admin/includes/plugin.php';
        require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';

        $remote_latest = isset($remote['latest']) ? (string) $remote['latest'] : '';
        $remote_name   = isset($remote['name']) ? (string) $remote['name'] : $slug;

        $package = '';
        if (!empty($remote['latest_package_url']) && $version !== '' && $version === $remote_latest) {
            $package = (string) $remote['latest_package_url'];
        }
        if ($package === '') {
            $package = $this->repo->package_url($slug, $version);
        }

        $installed_map = $this->installed_plugins_map();
        $candidates = isset($installed_map[$slug]) ? $installed_map[$slug] : [];
        $plugin_file = $this->pick_installed_plugin_file($slug, is_array($candidates) ? $candidates : [], $remote_name);

        $was_active = false;
        if ($plugin_file !== '' && function_exists('is_plugin_active')) {
            $was_active = is_plugin_active($plugin_file);
        }

        if ($was_active && $plugin_file !== '') {
            deactivate_plugins($plugin_file, false);
        }

        $fsr = $this->ensure_filesystem();
        if (!$fsr['ok']) {
            return ['ok' => false, 'msg' => $fsr['error']];
        }

        /** @var WP_Filesystem_Base $fs */
        $fs = $fsr['fs'];

        $plugin_dir = trailingslashit(WP_PLUGIN_DIR) . $slug;
        if ($fs->is_dir($plugin_dir)) {
            $deleted = $fs->delete($plugin_dir, true);
            if (!$deleted) {
                return ['ok' => false, 'msg' => 'Could not remove existing plugin folder before install/update.'];
            }
        }

        $skin = new Automatic_Upgrader_Skin();
        $upgrader = new Plugin_Upgrader($skin);

        ob_start();
        $result = $upgrader->install($package);
        ob_end_clean();

        if (is_wp_error($result)) {
            return ['ok' => false, 'msg' => 'Install failed: ' . $result->get_error_message()];
        }
        if ($result !== true) {
            return ['ok' => false, 'msg' => 'Install failed (unknown error).'];
        }

        wp_clean_plugins_cache(true);

        $installed_map2 = $this->installed_plugins_map();
        $candidates2 = isset($installed_map2[$slug]) ? $installed_map2[$slug] : [];
        $plugin_file2 = $this->pick_installed_plugin_file($slug, is_array($candidates2) ? $candidates2 : [], $remote_name);

        $should_activate = $activate_after || ($preserve_active && $was_active);

        if ($should_activate) {
            if ($plugin_file2 === '') {
                return ['ok' => true, 'msg' => 'Installed, but could not detect main plugin file to activate.'];
            }
            $r = activate_plugin($plugin_file2);
            if (is_wp_error($r)) {
                return ['ok' => true, 'msg' => 'Installed, but activation failed: ' . $r->get_error_message()];
            }
        }

        return ['ok' => true, 'msg' => 'OK'];
    }
}
